package edu.uky.ai.lp.logic;

/**
 * The superclass of all Boolean logical expressions.
 * 
 * @author Stephen G. Ware
 */
public abstract class BooleanExpression implements Expression {

	/** The name of the logical operator */
	public final String operator;
	
	/** The arguments to which the operator applies */
	protected final Expression[] arguments;
	
	/**
	 * Constructs a new Boolean expression with the given operator and
	 * arguments.
	 * 
	 * @param operator the Boolean operator to be used
	 * @param arguments the argument to which the operator applies
	 */
	public BooleanExpression(String operator, Expression[] arguments) {
		this.operator = operator;
		this.arguments = arguments;
	}
	
	@Override
	public boolean equals(Object other) {
		if(other instanceof BooleanExpression) {
			BooleanExpression otherBoolean = (BooleanExpression) other;
			if(!operator.equals(otherBoolean.operator) || arguments.length != otherBoolean.arguments.length)
				return false;
			for(int i=0; i<arguments.length; i++)
				if(!arguments[i].equals(otherBoolean.arguments[i]))
					return false;
			return true;
		}
		return false;
	}
	
	@Override
	public int hashCode() {
		int hc = operator.hashCode();
		for(Expression argument : arguments)
			hc += (argument.hashCode() * 31);
		return hc;
	}
	
	/**
	 * Given a separator character, this method returns a string representation
	 * of this expression with its argument separated.
	 * 
	 * @param separator the separator character
	 * @return a string
	 */
	protected String toString(String separator) {
		String str = "";
		for(Expression argument : arguments) {
			if(argument instanceof NAryBooleanExpression)
				str += separator + "(" + argument + ")";
			else
				str += separator + argument;
		}
		return str.substring(separator.length());
	}
	
	@Override
	public Unifier unify(Formula other, Unifier unifier) {
		if(other instanceof BooleanExpression) {
			BooleanExpression otherBoolean = (BooleanExpression) other;
			if(!operator.equals(otherBoolean.operator) || arguments.length != otherBoolean.arguments.length)
				return null;
			for(int i=0; i<arguments.length && unifier != null; i++)
				unifier = arguments[i].unify(otherBoolean.arguments[i], unifier);
			return unifier;
		}
		return null;
	}
	
	/**
	 * Returns the expression's arguments with their variables substituted.
	 * 
	 * @param unifier the unifier
	 * @return this expression's arguments, with variables substituted
	 */
	protected Expression[] substituteArguments(Unifier unifier) {
		Expression[] sub = new Expression[arguments.length];
		for(int i=0; i<sub.length; i++)
			sub[i] = arguments[i].substitute(unifier);
		return sub;
	}
}
